<?php
namespace App\Helpers;

use App\Models\DB;

function current_username(): string {
    $u = current_user();
    return $u['name'] ?? 'system';
}

/**
 * Audit helper that stores only changed fields with full before/after values.
 *
 * Behavior:
 *  - If $changes = ['before'=>array,'after'=>array], we emit only keys where values differ,
 *    with the full 'before' and full 'after' values.
 *  - If $changes has any other shape, it's logged as-is (backward compatible).
 *  - Column names in AuditLog are auto-detected to fit your schema.
 */
function audit(string $table, string $action, ?int $recordId, array $changes = []): void {
    $pdo  = DB::pdo();

    // ---- Compact to only changed fields (full values) when both sides provided ----
    if (isset($changes['before']) && is_array($changes['before']) &&
        isset($changes['after'])  && is_array($changes['after'])) {

        $before = $changes['before'];
        $after  = $changes['after'];

        $keys = array_unique(array_merge(array_keys($before), array_keys($after)));
        $beforeOut = [];
        $afterOut  = [];

        foreach ($keys as $k) {
            $bSet = array_key_exists($k, $before);
            $aSet = array_key_exists($k, $after);

            if ($bSet && !$aSet) {
                // Field removed
                $beforeOut[$k] = normalize_audit_value($before[$k]);
                continue;
            }
            if (!$bSet && $aSet) {
                // Field added
                $afterOut[$k]  = normalize_audit_value($after[$k]);
                continue;
            }

            // Present on both sides: include only if strict-different
            $bVal = normalize_audit_value($before[$k]);
            $aVal = normalize_audit_value($after[$k]);
            if ($bVal !== $aVal) {
                $beforeOut[$k] = $bVal;
                $afterOut[$k]  = $aVal;
            }
        }

        $changes = ['before' => $beforeOut, 'after' => $afterOut];
    }

    $user = $_SESSION['user']['name'] ?? ($_SESSION['user']['email'] ?? 'anonymous');
    $ip   = $_SERVER['REMOTE_ADDR']     ?? null;
    $ua   = $_SERVER['HTTP_USER_AGENT'] ?? null;
    $json = json_encode($changes, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    try {
        static $cols = null;
        if ($cols === null) {
            $cols = audit_detect_columns($pdo);
        }

        $columns = [$cols['TableName']];
        $values  = [$table];

        if ($cols['RecordId']    !== null) { $columns[] = $cols['RecordId'];    $values[] = $recordId; }
        if ($cols['Action']      !== null) { $columns[] = $cols['Action'];      $values[] = $action; }
        if ($cols['Changes']     !== null) { $columns[] = $cols['Changes'];     $values[] = $json; }
        if ($cols['PerformedBy'] !== null) { $columns[] = $cols['PerformedBy']; $values[] = $user; }
        if ($cols['IP']          !== null) { $columns[] = $cols['IP'];          $values[] = $ip; }
        if ($cols['UserAgent']   !== null) { $columns[] = $cols['UserAgent'];   $values[] = $ua; }

        // Optional time column with NOW(3)
        $timeColumnSql = '';
        if ($cols['PerformedAt'] !== null) {
            $columns[] = $cols['PerformedAt'];
            $timeColumnSql = 'NOW(3)';
        }

        $placeholders = array_fill(0, count($values), '?');
        if ($timeColumnSql !== '') {
            $sql = "INSERT INTO AuditLog (" . implode(', ', array_map(__NAMESPACE__ . '\\audit_quote_ident', $columns)) . ")
                    VALUES (" . implode(', ', $placeholders) . (count($placeholders) ? ', ' : '') . $timeColumnSql . ")";
        } else {
            $sql = "INSERT INTO AuditLog (" . implode(', ', array_map(__NAMESPACE__ . '\\audit_quote_ident', $columns)) . ")
                    VALUES (" . implode(', ', $placeholders) . ")";
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
    } catch (\Throwable $e) {
        \App\Helpers\log_error($e);
    }
}

/** Normalize typical values for strict comparison & JSON logging */
function normalize_audit_value($v) {
    if ($v instanceof \DateTimeInterface) return $v->format('Y-m-d H:i:s');
    return $v;
}

/** Inspect AuditLog and pick best-fit column names (cached). */
function audit_detect_columns(\PDO $pdo): array {
    $cols = [];
    $stmt = $pdo->query("SHOW COLUMNS FROM auditlog");
    foreach ($stmt->fetchAll(\PDO::FETCH_ASSOC) as $row) {
        $cols[strtolower($row['Field'])] = $row['Field'];
    }
    $pick = function(array $cands) use ($cols) {
        foreach ($cands as $c) {
            $k = strtolower($c);
            if (isset($cols[$k])) return $cols[$k];
        }
        return null;
    };
    $map = [
        'TableName'   => $pick(['TableName','Table','Entity','EntityName']),
        'RecordId'    => $pick(['RecordId','EntityId','RowId','RecordID']),
        'Action'      => $pick(['Action','Operation']),
        'Changes'     => $pick(['ChangesJson','Changes','Payload','Details','Data']),
        'PerformedBy' => $pick(['PerformedBy','User','Username','CreatedBy']),
        'PerformedAt' => $pick(['PerformedAt','CreatedAt','Timestamp','LoggedAt']),
        'IP'          => $pick(['IP','RemoteIP','ClientIP']),
        'UserAgent'   => $pick(['UserAgent','UA']),
    ];
    if ($map['TableName'] === null) {
        throw new \RuntimeException("AuditLog must have a 'TableName' column.");
    }
    return $map;
}

/** Backtick-quote identifiers for MySQL */
function audit_quote_ident(string $ident): string {
    return '`' . str_replace('`','``',$ident) . '`';
}

/**
 * Make a safe, readable filename base (keep letters, numbers, dashes, underscores).
 */
function slugify_filename_base(string $name): string {
    $name = preg_replace('/\.[^.]+$/', '', $name);                 // strip extension if any
    $name = iconv('UTF-8', 'ASCII//TRANSLIT', $name);
    $name = preg_replace('/[^A-Za-z0-9_\-]+/', '-', $name);
    $name = trim($name, '-_');
    return $name !== '' ? $name : 'file';
}

function ensure_dir(string $path): void {
    if (!is_dir($path) && !mkdir($path, 0775, true) && !is_dir($path)) {
        throw new RuntimeException("Failed to create directory: {$path}");
    }
}

/**
 * Find a collision-free filename with -n before the extension.
 * Uses an atomic 'x' create to avoid races.
 */
function next_available_name(string $dir, string $base, string $ext): string {
    $n = 0;
    while (true) {
        $suffix = $n === 0 ? '' : "-{$n}";
        $candidate = "{$base}{$suffix}" . ($ext ? ".{$ext}" : '');
        $full = "{$dir}/{$candidate}";

        // Atomic probe
        $h = @fopen($full, 'x'); // create if not exists, fail if exists
        if ($h !== false) {
            fclose($h);
            // We created an empty file to reserve the name; caller should overwrite/move into it.
            return $candidate;
        }
        $n++;
    }
}

/**
 * Store an uploaded file under uploads/YYYY/<tablename>/, with collision-safe name.
 * @return array{rel_path:string, url:string, disk_path:string, filename:string, mime:string, size:int}
 */
function store_uploaded_file(array $file, string $table, ?int $year = null): array {
    if (!isset($file['tmp_name'], $file['name']) || !is_uploaded_file($file['tmp_name'])) {
        throw new InvalidArgumentException('Invalid upload.');
    }

    $year = $year ?? (int)date('Y');
    // Normalize table name to a safe folder segment
    $tableSeg = preg_replace('/[^A-Za-z0-9_\-]/', '', $table);
    if ($tableSeg === '') {
        throw new InvalidArgumentException('Invalid table segment.');
    }

    $orig = $file['name'];
    $base = slugify_filename_base($orig);
    $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));

    $targetDir = UPLOAD_ROOT . "/{$year}/{$tableSeg}";
    ensure_dir($targetDir);

    // Reserve a unique name atomically
    $finalName = next_available_name($targetDir, $base, $ext);
    $diskPath  = "{$targetDir}/{$finalName}";

    // Move *into* the reserved file: since next_available_name created an empty file,
    // we replace it safely.
    if (!@rename($file['tmp_name'], $diskPath)) {
        // Fallback if rename fails across volumes
        if (!@move_uploaded_file($file['tmp_name'], $diskPath)) {
            @unlink($diskPath); // cleanup reserved
            throw new RuntimeException('Failed to move uploaded file.');
        }
    }

    // Detect mime & size
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime  = $finfo ? (finfo_file($finfo, $diskPath) ?: 'application/octet-stream') : 'application/octet-stream';
    if ($finfo) finfo_close($finfo);
    $size = filesize($diskPath) ?: 0;

    $rel  = "{$year}/{$tableSeg}/{$finalName}";
    $url  = rtrim(UPLOAD_BASE_URL, '/') . '/' . $rel;

    return [
        'rel_path'  => $rel,       // store this in DB
        'url'       => $url,       // render this in views
        'disk_path' => $diskPath,  // internal use
        'filename'  => $finalName,
        'mime'      => $mime,
        'size'      => $size,
    ];
}

