<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;
use function App\Helpers\url;
use function App\Helpers\handleupload;

use Exception;

class TaxMapsController
{
    /** GET /taxmaps */
    public static function index(): void
    {
        $pdo = DB::pdo();
 
        /* 0 (Tax Map) first, everything else second
           Extract year from the end of the Name field
           Make "Name" come before "Lot" */
        $sql = "SELECT Id, Name, Document from taxmaps
        ORDER BY
            SortOrder ASC, 
            CAST(SUBSTRING_INDEX(Name, '-', -1) AS UNSIGNED) DESC,
            CASE
                WHEN Name LIKE '%Name%' THEN 1
                WHEN Name LIKE '%Lot%'  THEN 2
                ELSE 3
            END ASC;
        ";
        $rows = $pdo->query($sql)->fetchAll();

        $page_title = 'Tax Map';
        require __DIR__ . '/../Views/taxmaps/list.php';
    }

    /** GET /taxmaps/create */
    public static function create(): void
    {
        $pdo = DB::pdo();

        $record = [
            'Id' => null,
            'Name' => '',
            'Document' => ''
        ];
        $page_title = 'Add Tax Map';
        require __DIR__ . '/../Views/taxmaps/form.php';
    }

    /** POST /taxmaps/store */
    public static function store(): void
    {
        csrf_verify();
        $pdo = DB::pdo();
        $createdBy = \App\Helpers\current_username();

        $Name = trim($_POST['Name'] ?? null);
        $Document = trim($_POST['Document'] ?? null);

        if (!$Name || !$Document) {
            http_response_code(400);
            echo "Name and Document are required.";
            return;
        }

        //Save file/url
        $parentTable = 'taxmaps';
        $upload = \App\Helpers\handleupload('attachment_Document', $parentTable);
        if ($upload) {
            // Build full URL using config PrependURL
            $Document = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
        }

        $sql = "
            INSERT INTO taxmaps
            (Name, Document, CreatedAt, CreatedBy)
            VALUES (?, ?, NOW(), ?)
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $Name,
            $Document,
            $createdBy
        ]);

        $id = (int)$pdo->lastInsertId();
        audit('TaxMaps','create',$id,[
            $Name,
            $Document,
            $createdBy
        ]);
        redirect('/taxmaps');
        //header("Location: /taxmaps");
        //exit;
    }

    /** GET /taxmaps/edit?id=123 */
    public static function edit(): void
    {
        $id = (int)($_GET['id'] ?? 0);
        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }

        $pdo = DB::pdo();

        $stmt = $pdo->prepare("SELECT * FROM taxmaps WHERE Id = ?");
        $stmt->execute([$id]);
        $model = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$model) {
            http_response_code(404);
            echo "TaxMap record not found.";
            return;
        }

        $errors = [];
        ob_start();
        $page_title = 'Edit Tax Map';
        include __DIR__ . '/../Views/taxmaps/form.php';
        //return ob_get_clean();
    }

    /** POST /taxmaps/update */
    public static function update(): void
    {
        csrf_verify();
        $id  = (int)($_POST['Id'] ?? 0);
        $pdo = DB::pdo();

        $sel = $pdo->prepare("SELECT * FROM taxmaps WHERE Id=?");
        $sel->execute([$id]);
        $before = $sel->fetch() ?: [];
        //\App\Helpers\log_info("before uploada: {$sel['Name']}");
        //\App\Helpers\log_info("before uploada: {$sel['Document']}");

        $Name = trim($_POST['Name'] ?? null);
        \App\Helpers\log_info("before uploadb: {$Name}");
        \App\Helpers\log_info("before upload2: {$_POST['Document']}");
        $Document = trim($_POST['Document'] ?? null);
        \App\Helpers\log_info("before uploadb: {$Document}");

        if (!$id || !$Name || !$Document) {
            http_response_code(400);
            echo "Id, Name and Document are required.";
            return;
        }

        //Save file/url
        \App\Helpers\log_info("before upload: {$Document}");
        $parentTable = 'taxmaps';
        //\App\Helpers\log_info('FILES: ' . print_r($_FILES, true));
        $upload = \App\Helpers\handleupload('attachment_Document', $parentTable);
        if ($upload) {
            // Build full URL using config PrependURL
            $Document = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
        }
        \App\Helpers\log_info("after upload: {$Document}");

        $sql = "
            UPDATE taxmaps
            SET Name=?, Document=?
            WHERE Id=?
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $Name,
            $Document,
            $id
        ]);

        $sel->execute([$id]);
        $after = $sel->fetch() ?: [];
        audit('TaxMaps','update',$id,['before'=>$before,'after'=>$after]);
        redirect('/taxmaps');
        //header("Location: /taxmaps");
        //exit;
    }

    /** POST /taxmaps/delete */
    public static function delete(): void
    {
        $id = (int)($_POST['id'] ?? 0);
        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }

        $pdo = DB::pdo();
        $sel = $pdo->prepare("SELECT * FROM taxmaps WHERE Id=?");
        $sel->execute([$id]);
        $row = $sel->fetch() ?: [];

        $stmt = $pdo->prepare("DELETE FROM taxmaps WHERE Id = ?");
        $stmt->execute([$id]);
        audit('TaxMaps','delete',$id,['before'=>$row]);

        redirect('/taxmaps');
        //header("Location: /taxmaps");
        //exit;
    }
}
