<?php
// Public Site Entry (no auth). Uses existing config + DB model.
// URL: /WebSite/index.php

declare(strict_types=1);

// Bootstrap config and DB
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/app/Models/DB.php';

use App\Models\DB;
#use PDO;

// ---------- BASE URL (safe on Windows/IIS) ----------
$scriptName = isset($_SERVER['SCRIPT_NAME']) ? $_SERVER['SCRIPT_NAME'] : '';
$BASE_URL   = rtrim(dirname($scriptName), "/\\");
if ($BASE_URL === '' || $BASE_URL === '/' || $BASE_URL === '\\') { $BASE_URL = ''; }
function url($path = '') {
  global $BASE_URL; $path = ltrim($path, '/');
  return ($BASE_URL ? $BASE_URL.'/' : '/').$path;
}

// ---------- HELPERS ----------
function esc($s){ return htmlspecialchars((string)($s ?? ''), ENT_QUOTES, 'UTF-8'); }
function body_center($html){ return '<div class="text-center">'.$html.'</div>'; }
function hbold($text){ return '<strong>'.esc($text).'</strong>'; }

function build_file_href($filePath, $fileName) {
  if (!$fileName) return null;
  $path = rtrim((string)$filePath, '/');
  $name = ltrim((string)$fileName, '/');
  $joined = ($path ? ($path.'/'.$name) : $name);
  if (preg_match('~^https?://~i', $joined)) return $joined;
  return url(trim($joined, '/'));
}

function render_attachments($atts) {
  if (!$atts) return '';
  $out = '<ul class="list-unstyled ms-3 mb-0">';
  foreach ($atts as $a){
    $label = 'Attachment';
    if (is_array($a) && isset($a['Title']) && $a['Title'] !== '') {
      $label = $a['Title'];
    }
    $label = esc($label);

    $href = null;
    if (isset($a['Url']) && $a['Url'] !== '') {
      $href = $a['Url'];
    } else {
      $fp = isset($a['FilePath']) ? $a['FilePath'] : null;
      $fn = isset($a['FileName']) ? $a['FileName'] : null;
      $href = build_file_href($fp, $fn);
    }

    if ($href) {
      $out .= '<li>• <a href="'.esc($href).'" target="_blank" rel="noopener">'.$a['Title'].'</a></li>';
    } else {
      $out .= '<li>• '.$label.'</li>';
    }
  }
  $out .= '</ul>';
  return $out;
}

function fetch_attachments(PDO $pdo, $parentTable, $parentIds) {
  if (!$parentIds) return [];
  $in  = implode(',', array_fill(0, count($parentIds), '?'));
  $sql = "SELECT AttachmentId, ParentId, Title, Url, FilePath, FileName
          FROM Attachments
          WHERE IsActive = 1 AND ParentTable = ? AND ParentId IN ($in)
          ORDER BY AttachmentId";
  $stmt = $pdo->prepare($sql);
  $params = array_merge([$parentTable], $parentIds);
  $stmt->execute($params);
  $by = [];
  while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $by[(int)$row['ParentId']][] = $row;
  }
  return $by;
}

// ---------- QUERY DATA ----------
try {
  $pdo = DB::pdo();

  // Town (active)
  $town = $pdo->query("SELECT * FROM Town WHERE Active = 1 ORDER BY Id LIMIT 1")->fetch();
  if (!$town) {
    $town = [
      'Name' => 'Town of Cushing, Maine',
      'Address1' => '39 Cross Road', 'Town' => 'Cushing', 'State' => 'ME', 'Zip' => '04563',
      'Phone1' => '(207) 354-2375', 'Remarks' => null
    ];
  }

  // Announcements: view if present, else fallback
  $annSql = "SELECT Id, Title, Body, StartDate, EndDate
             FROM Announcements
             WHERE IsActive = 1
               AND (StartDate IS NULL OR StartDate <= UTC_DATE())
               AND (EndDate   IS NULL OR EndDate   >= UTC_DATE())
             ORDER BY StartDate DESC, Id DESC";
  $ann = $pdo->query($annSql)->fetchAll();
  $annIds = array_map(function($r){ return (int)$r['Id']; }, $ann);
  $annAtt = fetch_attachments($pdo, 'Announcements', $annIds);

  // Boards
  $boards = $pdo->query("SELECT Id, CommitteeName AS Name FROM BoardsAndCommittees WHERE Active = 1 ORDER BY CommitteeName")->fetchAll();

  // Community
  $community = $pdo->query("SELECT * from Community ORDER BY Name")->fetchAll();

  // Events
  #$evtSql = "SELECT e.Id, e.Title, e.Body, e.Location, e.StartDateTime, e.EndDateTime, e.AllDay, e.BoardId, b.CommitteeName
  $evtSql = "SELECT e.Id, e.Title, e.Body, e.Location, e.StartDateTime, e.EndDateTime, e.AllDay, e.BoardId
             FROM Events e
             #LEFT JOIN BoardsAndCommittees b ON b.Id = e.BoardId
             #WHERE e.StartDateTime >= UTC_TIMESTAMP()
             ORDER BY e.StartDateTime ASC";
  $events = $pdo->query($evtSql)->fetchAll();
  $evtIds = array_map(function($r){ return (int)$r['Id']; }, $events);
  $evtAtt = fetch_attachments($pdo, 'Events', $evtIds);

  // Fire and Safety
  $fireandsafety = $pdo->query("SELECT * from FireAndSafety ORDER BY Category")->fetchAll();

  // Information
  $information = $pdo->query("SELECT * from Information ORDER BY Category, Role")->fetchAll();

  // Ordinances
  $ordinances = $pdo->query("SELECT Id, Category, Title, Summary FROM OrdinancesRegulationsForms ORDER BY FIELD(Category, 'Ordinances', 'Regulations', 'Forms', 'Miscellaneous'), Title")->fetchAll();

  // Resources
  $resSql = "SELECT Id, Title, Url, Description, Category, SortOrder
             FROM Resources
             WHERE Active = 1
             ORDER BY SortOrder, Id";
  $resources = $pdo->query($resSql)->fetchAll();
  $resIds = array_map(function($r){ return (int)$r['Id']; }, $resources);
  $resAtt = fetch_attachments($pdo, 'Resources', $resIds);

  // TaxMaps
  #$taxmaps = $pdo->query("SELECT Name, Document FROM TaxMaps ORDER BY Name")->fetchAll();
  $taxSql = "SELECT Name, Document FROM TaxMaps ORDER BY
    SortOrder ASC,       -- 0 (Tax Map) first, everything else second
    CAST(SUBSTRING_INDEX(Name, '-', -1) AS UNSIGNED) DESC,
    CASE
        WHEN Name LIKE '%Name%' THEN 1
        WHEN Name LIKE '%Lot%'  THEN 2
        ELSE 3
    END ASC";
  $taxmaps = $pdo->query($taxSql)->fetchAll();

  // Meetings and Reports
  $townmeetingsandreports = $pdo->query("SELECT * from TownMeetingsAndReports ORDER BY Year desc")->fetchAll();

  // Staff
  $townstaffpositions = $pdo->query("SELECT * from TownStaffPositions ORDER BY JobTitle")->fetchAll();

// -- Boards & Committees with subordinates and minutes
$sql = "
SELECT 
  b.Id AS BoardId,
  b.CommitteeName,
  b.MeetingScheduleModel,
  b.Email,

  mm.Id AS MeetingId,
  mm.MeetingDate AS MeetingDate,
  mm.AgendaUrl AS AgendaUrl,

  mem.PersonName,
  op.Name AS OfficerPosition
FROM BoardsAndCommittees b
LEFT JOIN BoardMeetingMinutes mm ON mm.BoardId = b.Id
LEFT JOIN BoardsAndCommitteeMembers mem ON mem.BoardId = b.Id
LEFT JOIN OfficerPositions op ON op.Id = mem.OfficerPositionId
WHERE b.Active = 1
ORDER BY b.CommitteeName, mm.MeetingDate DESC, mem.PersonName
";

$rows = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

// Group the hierarchical structure in PHP
$tboards = [];
foreach ($rows as $r) {
    $bid = $r['BoardId'];
    if (!isset($tboards[$bid])) {
        $tboards[$bid] = [
            'CommitteeName' => $r['CommitteeName'],
            'MeetingScheduleModel' => $r['MeetingScheduleModel'],
            'Email' => $r['Email'],
            'Members' => [],
            'Meetings' => []
        ];
    }

    if ($r['PersonName']) {
        $tboards[$bid]['Members'][] = [
            'PersonName' => $r['PersonName'],
            'OfficerPosition' => $r['OfficerPosition']
        ];
    }

    if ($r['MeetingId']) {
        if (!isset($tboards[$bid]['Meetings'][$r['MeetingId']])) {
            $tboards[$bid]['Meetings'][$r['MeetingId']] = [
                'MeetingDate' => $r['MeetingDate'],
                'AgendaUrl' => $r['AgendaUrl'],
                'Minutes' => []
            ];
        }
    }
}


} catch (\Throwable $e) {
  http_response_code(500);
  echo '<pre style="padding:16px;background:#fff0f0;border:1px solid #f9c2c2;border-radius:8px;">DB connection/query failed: '.esc($e->getMessage())."</pre>";
  exit;
}

?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?= esc(isset($town['Name']) ? $town['Name'] : 'Town of Cushing, Maine') ?></title>
  <link rel="stylesheet" href="./style.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css">
  <style>
    :root{
      --ink:#1f2937; --muted:#6b7280; --brand:#15344a; --accent:#8a0707; --card:#ffffff;
    }
    body{margin:0; font:16px/1.5 system-ui,Segoe UI,Roboto,Helvetica,Arial; color:var(--ink); background:#fff}
    a{color:var(--brand); text-decoration:none}
    a:hover{text-decoration:underline}
    .brand-hr{border:0;border-top:4px solid var(--accent);opacity:1;margin:.75rem auto;max-width:200px}
    .card{border-radius:14px}
    .muted{color:var(--muted)}
    html { scroll-behavior: smooth; scroll-padding-top: var(--header-h); }
  </style>
</head>
<body class="bg-light">

<!-- ===== HEADER IMAGE (Auto Directory Scan) ===== -->
<?php
// Read image filenames from /headerimages
$headerDir = __DIR__ . "/headerimages";
$headerUrl = "/WebSite/headerimages"; // adjust if needed

$files = [];
if (is_dir($headerDir)) {
    foreach (scandir($headerDir) as $f) {
        if (preg_match('/\.(png|jpg|jpeg|webp)$/i', $f)) {
            $files[] = $f;
        }
    }
}
sort($files); // alphabetical
?>

<style>
.email-link {
    color: #0d6efd !important; /* Bootstrap primary blue */
    text-decoration: underline !important;
}

.site-header {
    position: relative;
    width: 100%;      /* ensures full-screen width */
    height: 240px;    /* header height */
    overflow: hidden;
}

/* Apply to ALL slider images */
.headerImage {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;       /* stretch horizontally */
    height: 100%;      /* stretch vertically */
    object-fit: fill;  /* ⭐ THIS forces distortion */
    display: block;
}
</style>

<div id="fixedTop">

<header class="site-header">

    <!-- MAIN IMAGE -->
    <img class="headerImage"
         id="headerImage"
         src="<?= $headerUrl . '/' . ($files[0] ?? '') ?>"
         alt="Cushing Maine header">

    <script>
    (function() {

        const images = <?= json_encode($files) ?>;
        const folder  = "<?= $headerUrl ?>/";

        if (images.length < 2) return; // no need for slider

        const intervalSeconds = 30;
        const transitionMs = 800;

        const header  = document.querySelector('header');
        let current   = document.getElementById('headerImage');

        // Create clone for sliding transition
        let next = current.cloneNode();
        next.classList.add("headerImage");   // IMPORTANT
        next.style.opacity = 0;
        header.appendChild(next);

        let index = 0;

        function slideToNext() {
            index = (index + 1) % images.length;
            const newSrc = folder + images[index];

            // preload target image
            const preload = new Image();
            preload.onload = () => {
                next.style.transition = "none";
                next.style.transform  = "translateX(100%)";
                next.style.opacity    = 1;
                next.src              = preload.src;

                void next.offsetWidth; // reflow

                // slide animation
                current.style.transition = next.style.transition =
                    `transform ${transitionMs}ms ease-in-out`;

                current.style.transform = "translateX(-100%)";
                next.style.transform    = "translateX(0)";

                setTimeout(() => {
                    // swap elements
                    const temp = current;
                    current = next;
                    next    = temp;

                    // reset & ensure BOTH images keep correct class
                    next.style.transition = "none";
                    next.style.transform  = "translateX(0)";
                    next.style.opacity    = 0;
                    next.classList.add("headerImage");
                    current.classList.add("headerImage");

                }, transitionMs);
            };

            preload.src = newSrc;
        }

        setInterval(slideToNext, intervalSeconds * 1000);

        // Apply transition rule to all header images
        const style = document.createElement("style");
        style.textContent = `
          header img {
            transition: transform ${transitionMs}ms ease-in-out;
          }
        `;
        document.head.appendChild(style);

    })();
    </script>
</header>
  
  <section class="pb-0 bg-body-tertiary text-center">
    <div class="container">
      <h1 class="h3 mb-1"><p style="font-family: Oswald;">Welcome to the <?= esc(isset($town['Name']) ? $town['Name'] : 'Cushing, Maine') ?></p></h1>
      <h2 class="h3 mb-1"><p style="font-family: Oswald;">ONE OF MAINE'S UNIQUE COASTAL COMMUNITIES ESTABLISHED 1789</p></h2>
    </div>
  </section>

  <!-- ===== NAVBAR (top) ===== -->
  <nav class="navbar navbar-expand-lg bg-light border-bottom shadow-sm">
    <div class="container-fluid">
      <!--a class="navbar-brand fw-bold" href="/">Town of Cushing</a-->
      <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#mainNav">
        <span class="navbar-toggler-icon"></span>
      </button>
      <div class="collapse navbar-collapse" id="mainNav">
        <ul class="navbar-nav me-auto mb-2 mb-lg-0">
          <?php
          $tabs=['announcements'=>'Announcements','boards'=>'Boards & Committees','community'=>'Community','events'=>'Events','fireandsafety'=>'Fire/Safety','information'=>'Information','ordinances'=>'Ordinances/Regulations/Forms',
                'resources'=>'Resources','taxmaps'=>'Tax Maps', 'townmeetingsandreports'=>'Town Meetings/Reports', 'townstaffpositions'=>'Town Staff'];
          foreach($tabs as $id=>$label):?>
          <li class="nav-item">
            <a class="nav-link" href="#section-<?=$id?>"><?=$label?></a>
          </li>
          <?php endforeach;?>
        </ul>
      </div>
    </div>
  </nav>
</div>

<div id="scrollArea">
  <main class="container my-4">
    <?php
    if (isset($_GET['townstaffid']) && ctype_digit($_GET['townstaffid'])) {
      // ===== HARD-CODED DETAIL VIEW FOR TOWN STAFF ONLY =====
      include __DIR__ . "/townstaffpositions/show.php";
    } else {
      // full page render
      // Load all tab content on page load
      foreach (['announcements','boards','community','events','fireandsafety','information','ordinances','resources','taxmaps','townmeetingsandreports','townstaffpositions'] as $id) {
          echo "<div id='section-$id' class='mb-5'>";
          include __DIR__ . "/partials/$id.php";
          echo "</div>";
      }
    }
    //    return; // stop rendering the rest of the index page body
    ?>
  </main>

  <footer class="bg-body-tertiary py-3">
    <div class="container text-center text-muted small">
      <?php
        $addr = trim((isset($town['Address1'])?$town['Address1']:'').' '.(isset($town['Town'])?$town['Town']:'').', '.(isset($town['State'])?$town['State']:'').' '.(isset($town['Zip'])?$town['Zip']:''));
        $phone = isset($town['Phone1']) ? $town['Phone1'] : '';
      ?>
      <div><?= esc(isset($town['Name']) ? $town['Name'] : 'Town of Cushing, Maine') ?> · <?= esc($addr) ?> · <?= esc($phone) ?></div>
      <div>&copy; <?= date('Y') ?> <?= esc(isset($town['Name']) ? $town['Name'] : 'Town of Cushing, Maine') ?>. All rights reserved.</div>
    </div>
  </footer>
</div>

<-- scripts -->
<script>
// Understand the components of the fixed part of the screen
(function () {
  function setOffsets() {

    const header = document.querySelector('#fixedTop > header');
    const welcome = document.querySelector('#fixedTop > section');
    const navbar = document.querySelector('#fixedTop > nav');

    if (!header || !welcome || !navbar) {
      console.warn("Missing header, welcome section, or navbar");
      return;
    }

    const h =
      header.offsetHeight +
      welcome.offsetHeight +
      navbar.offsetHeight;

    console.log("Setting --header-h to", h + "px");

    document.documentElement.style.setProperty('--header-h', h + 'px');
  }

  window.addEventListener('load', () => setTimeout(setOffsets, 50));
  window.addEventListener('resize', setOffsets);

  // ensure header height is correct when header image finishes loading
  const hdrImg = document.getElementById('headerImage');
  if (hdrImg) hdrImg.addEventListener('load', () => setTimeout(setOffsets, 50));
})();
</script>

<script>
  // Toggle when the ellipses is hit so it scrolls to the top
  document.addEventListener('click', function (e) {
    const el = e.target.closest('.clamp-3');
    if (el) {
      const wasExpanded = el.classList.contains('expanded');
      el.classList.toggle('expanded');

      // if just expanded, scroll to keep it nicely positioned
      if (!wasExpanded) {
        const card = el.closest('.card');
        const container = document.getElementById('scrollArea') || window;
        if (card) {
          card.scrollIntoView({
            behavior: 'smooth',
            block: 'start'
          });
        } else {
          el.scrollIntoView({
            behavior: 'smooth',
            block: 'start'
          });
        }
      }
    }
  });
</script>

<script>
//A hack to get a const of how far down the actual 1st section is from the top
//const targetTop = document.getElementById('section-announcements')
//const firstContentTop = targetTop.querySelector('h2');
const gap = 24; //firstContentTop ? firstContentTop.offsetHeight : 0;
console.log("gap=" + gap)

document.addEventListener('DOMContentLoaded', () => {
  const scrollArea = document.getElementById('scrollArea');
  console.log("scrollArea=" + scrollArea)
  const fixedTop = document.getElementById('fixedTop');
  console.log("fixedTop=" + fixedTop)

  document.querySelectorAll('.navbar .nav-link[href^="#"]').forEach(link => {
    link.addEventListener('click', function (e) {
      e.preventDefault();

      const targetId = this.getAttribute('href').substring(1);
      console.log("targetId=" + targetId)
      const target = document.getElementById(targetId);
      if (!target) return;
      console.log("target=" + target)

      const fixedOffset = fixedTop ? fixedTop.offsetHeight : 0;
      console.log("fixedOffset=" + fixedOffset)

      // target position inside scrollArea
      const rect = target.getBoundingClientRect();
      console.log("rect=" + rect)
      const targetY = rect.top + scrollArea.scrollTop - fixedOffset;
      console.log("targetY=" + targetY)

      scrollArea.scrollTo({ top: targetY - gap, behavior: 'auto' });
      console.log("main scrolled to=" + (targetY - gap))
    });
  });
});
</script>

<script>
(function () {

  // --------------------------
  // SAVE SCROLL POSITION
  // --------------------------
  document.addEventListener('click', function(e) {
    const link = e.target.closest('a');
    if (!link) return;

    // DO NOT save scrollPos while in detail page — prevents overwriting with "0"
    if (window.location.search.includes("townstaffid=")) {
      console.log("Skipping scrollPos save in detail mode");
      return;
    }

    const scrollArea = document.getElementById('scrollArea');
    if (!scrollArea) return;

    const pos = scrollArea.scrollTop;
    localStorage.setItem('scrollPos', pos);
    console.log("scrollPos saved: " + pos);
  });

  // --------------------------
  // RESTORE SCROLL POSITION
  // --------------------------
  //window.addEventListener('DOMContentLoaded', () => {
  window.addEventListener('load', () => {

    // If ON the detail page → skip restore (cannot scroll)
    if (window.location.search.includes("townstaffid=")) {
      console.log("*** detail page detected — skip scroll restore ***");
      return;
    }

    // If we just came back from a pending nav (e.g. Community from detail),
    // skip restoring the old scrollPos so the nav target wins.
    if (sessionStorage.getItem("skipScrollRestore") === "1") {
      console.log("*** skipScrollRestore flag set — skipping scroll restore ***");
      sessionStorage.removeItem("skipScrollRestore");
      return;
    }

    // If we have a pending navigation jump, DO NOT restore scroll
    //if (sessionStorage.getItem("pendingSection")) {
    //    console.log("*** pending nav exists — skip scroll restore ***");
    //    return;
    //}

    const scrollArea = document.getElementById('scrollArea');
    if (!scrollArea) return;

    const pos = localStorage.getItem('scrollPos');
    console.log("pos from storage: " + pos);
    if (pos !== null) {
      localStorage.removeItem('scrollPos');
      scrollArea.scrollTo({ top: parseInt(pos, 10), behavior: 'instant' });
      console.log("restored scroll to=" + parseInt(pos, 10));
    }
  });

})();
</script>

<script>
document.addEventListener("DOMContentLoaded", () => {
  // A test where we are loading town staff in the lower scrollable section
  console.log("*** entering Town Staff item ***");
  const navLinks = document.querySelectorAll(".navbar a.nav-link");
  //console.log("Navbar links found:", document.querySelectorAll(".navbar a.nav-link").length);

  navLinks.forEach(link => {
    link.addEventListener("click", function(e) {
      const hasQuery = window.location.search.length > 0;
      console.log("hasQuery=" + hasQuery)

      if (hasQuery) {
        e.preventDefault();  // stop the jump
        const cleanUrl = window.location.pathname; // e.g., /WebSite/index.php
        console.log("cleanUrl=" + cleanUrl)

        // store which section was clicked
        sessionStorage.setItem("pendingSection", this.getAttribute("href"));

        // reload clean
        window.location.href = cleanUrl;
      }
    });
  });

  // After reload: was a nav click pending?
  const pending = sessionStorage.getItem("pendingSection");
  console.log("pending1=" + pending)
  if (pending) {
    console.log("*** a nav click was pending ***");

    // Tell the restore logic to skip restoring old scrollPos this time
    sessionStorage.setItem("skipScrollRestore", "1");

    sessionStorage.removeItem("pendingSection");
    console.log("pending2=" + pending)

    // Jump to the section the user originally clicked
    if (pending.startsWith("#")) {
      console.log("*** pending started with # ***");
      const target = document.querySelector(pending);
      console.log("target=" + target)
      //if (target) {

if (target) {

    // Match the normal navbar scrolling behavior
    const scrollArea = document.getElementById('scrollArea');
    const fixedTop = document.getElementById('fixedTop');

    const rect = target.getBoundingClientRect();
    const fixedOffset = fixedTop ? fixedTop.offsetHeight : 0;
    const targetY = rect.top + scrollArea.scrollTop - fixedOffset;

    scrollArea.scrollTo({
        top: targetY - 24,   // gap = 24 (matching your existing behavior)
        behavior: "auto"     // instant jump, not smooth
    });
}


        //target.scrollIntoView({ behavior: "smooth" });
      //}
    }
  }
  else
  {
    console.log("*** a nav click was not pending ***");
  }
  console.log("*** leaving Town Staff item ***");
});
</script>

<?php
if (isset($_GET['townstaffid']) && ctype_digit($_GET['townstaffid'])) {
?>
<script>
//scroll to the top after loading our test town staff in the scrollable section of the screen
document.addEventListener("DOMContentLoaded", function() {
    console.log("Detail page loaded (no forced scroll)");
});
</script>
<?php
} 
?>

<!-- Bootstrap JS bundle (includes Popper) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

</body>
</html>
