import os
import time
import mysql.connector
from datetime import datetime
from pathlib import Path

from docx import Document
from docx.shared import Inches

from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.support.ui import WebDriverWait, Select
from selenium.webdriver.support import expected_conditions as EC
from webdriver_manager.chrome import ChromeDriverManager

DEBUG = True

#pick a good BoardId after you have any in the table
BOARD_ID = 74
BOARD_NAME = 'TEST'

#if you want to save your creates and edits, set to false 
DELETE_ALL = True

def set_board_id():
    global BOARD_ID
    global BOARD_NAME

    cnx = mysql.connector.connect(
        host="localhost",      # or remote host
        user="root",           # same credentials as PHP
        password="L@rryEll1s0n!$R1ch",
        database="townofcushingme"
    )

    cursor = cnx.cursor(dictionary=True)
    cursor.execute("SELECT Id, CommitteeName FROM boardsandcommittees ORDER BY Id DESC LIMIT 1")
    row = cursor.fetchone()

    if row:
        BOARD_ID = row["Id"]
        BOARD_NAME = row["CommitteeName"]
        print("Last BoardId =", BOARD_ID)
        print("Last CommitteeName =", BOARD_NAME)
    else:
        print("No records found")

    cursor.close()
    cnx.close()

def log(msg):
    if DEBUG:
        print(f"[{time.strftime('%H:%M:%S')}] {msg}")

# ----------------------------
# CONFIG — CHANGE THESE
# ----------------------------
BASE_URL = "http://localhost/maintenance/public/index.php"
#BASE_URL = "http://127.0.0.1/maintenance/public/index.php"
#BASE_URL = "http://DESKTOP-SE84HIF/maintenance/public/index.php"

# if you have auth, set LOGIN_URL and implement login() below
#LOGIN_URL = None  # e.g., f"{BASE_URL}/login"
LOGIN_URL = f"{BASE_URL}"

# Minimal field mappings per module. Keys are HTML "name" attributes in your forms.
# "link_field" is the column used as the anchor link on the list page.
MODULES = {
    "announcements": {
        # use the exact input names from your form
        "create": {
            "Title": "Test Announcement",
            "Body": "Automated test body created by Selenium."
            # Dates are injected at runtime (see create_record) to keep formats correct
        },
        "edit_append": "edited",
        "link_field": "Title",  # the linked column on the list page
    },
    "boardsandcommittees": { # make sure this is always before the other boardxxxxxx tables
        "create": {"CommitteeName": "New Committee",
                   "Description": "Test Desc",
                   "Email": "test@aol.com",
                   "BylawsUrl": "http://dave.com",
                   "Remarks": "Test Remarks"
        },
        "edit_append": "edited",
        "link_field": "CommitteeName",
    },
    #"boardlinks": {
    #    "create": {"Title": "QA Title", "Url": "https://cushing.maine.gov"},
    #    "edit_append": " (edited)",
    #    "link_field": "Title",
    #},
    #"boardmeetings": {
    #    "create": {"Notes": "Test Notes",
    #               "AgendaUrl": "http://dave.com",
    #               "MinutesUrl": "http://dave.com",
    #               "RecordingUrl": "http://dave.com"
    #    },
    #    "edit_append": "edited",
    #    "link_field": "Notes",
    #},
    "boardmeetings": {
        "create": {"AgendaUrl": "http://dave.com",
                   "ApprovedMinutesUrl": "http://dave.com",
                   "RecordingUrl": "http://dave.com"
         },
        "edit_append": "edited",
        "link_field": "AgendaUrl",
    },
    "boardsandcommitteemembers": {
        "create": {"PersonName": "TestPerson",
                   "Address1": "TestAddress",
                   "Town": "Anywhere",
                   "State": "ME",
                   "Zip": "99999",
                   "Phone1": "207-222-0000",
                   "Email": "test@gmail.com"
        },
        "edit_append": "edited",
        "link_field": "PersonName",
    },
    #"boardschedulerules": {
    #    "create": {"HolidayPolicy": "HolidayPolicy",
    #               "Notes": "TestNotes"
    #},
    #    "edit_append": "edited",
    #    "link_field": "HolidayPolicy",
    #},
    "community": {
        "create": {"Name": "TestCommunity", 
                   "Category": "Other"
        },
        "edit_append": "edited",
        "link_field": "Name",
    },
    "events": {
        "create": {"Title": "TestTitle",
                   "Body": "TestBody",
                   "Location": "TestLoc"
        },
        "edit_append": "edited",
        "link_field": "Title",
    },
    "fireandsafety": {
        "create": {"Category": "TestCategory",
                   "Position": "TestPosition"
        },
        "edit_append": "edited",
        "link_field": "Category",
    },
    "information": {
        "create": {"Category": "New Category", 
                   "Role": "TestRole",
                   "ContactName": "TestContact",
                   "Location": "TestLoc",
                   "MailAddress": "TestAddr",
                   "PhonePrimary": "207-222-0000",
                   "Email": "test@google.com",
                   "Notes": "TestNotes"
        },
        "edit_append": "edited",
        "link_field": "Role",
    },
    #"officerpositions": {
    #    "create": {"Name": "OfficerPosition"
    #    },
    #    "edit_append": "edited",
    #    "link_field": "Name",
    #},
    "ordinancesregulationsforms": {
        "create": {"Category": "Miscellaneous", 
                   "Title": "http://www.cushing.maine.gov/wp-content/uploads/2024/09/The-Cushing-Addressing-Ordinance-1996.pdf",
                   "Summary": "TestSummary"
        },
        "edit_append": "edited",
        "link_field": "Title",
    },
    "resources": {
        "create": {"Title": "Test Title",
                   "Url": "http://www.cushing.maine.gov/wp-content/uploads/2024/09/The-Cushing-Addressing-Ordinance-1996.pdf",
                   "Description": "TestDesc",
                   "Category": "TestCategory",
                   "Remarks": "TestRemarks"
        },
        "edit_append": "edited",
        "link_field": "Title",
    },
    "taxmaps": {
        "create": {"Name": "TestName", 
                   "Document": "http://www.cushing.maine.gov/wp-content/uploads/2024/09/The-Cushing-Addressing-Ordinance-1996.pdf"
        },
        "edit_append": "edited",
        "link_field": "Name",
    },
    "townmeetingsandreports": {
        "create": {"Year": "2000",
                   "WarrantUrl": "http://dave.com"
        },
        "edit_append": "edited",
        "link_field": "Year",
    },
    "townstaffpositions": {
        "create": {"JobTitle": "TestJob", 
                   "PersonName": "Austin",
                   "Address1": "TestAddr1",
                   "Town": "Anywhere",
                   "State": "ME",
                   "Zip": "99999",
                   "Phone1": "207-222-0000",
                   "Email": "test@google.com",
                   "Remarks": "TestRemarks"
        },
        "edit_append": "edited",
        "link_field": "JobTitle",
    },
}

# Optional: default active checkbox name used in most forms
ACTIVE_CHECKBOX = "IsActive"

# Screenshot directory + doc output
RUN_STAMP = datetime.now().strftime("%Y%m%d_%H%M%S")
OUT_DIR = Path(f"crud_test_run_{RUN_STAMP}")
OUT_DIR.mkdir(parents=True, exist_ok=True)

DOC = Document()
DOC.add_heading("CRUD Flow Test & User Guide Shots", 0)

def get_form_element(driver, name):
    """Return the form element by name, or None if not found."""
    try:
        return driver.find_element(By.NAME, name)
    except Exception:
        return None

def is_select_element(el):
    try:
        return el is not None and el.tag_name.lower() == "select"
    except Exception:
        return False

def start_browser():
    options = webdriver.ChromeOptions()
    options.add_argument("--start-maximized")
    # options.add_argument("--headless=new")  # uncomment for headless
    driver = webdriver.Chrome(service=Service(ChromeDriverManager().install()), options=options)
    driver.set_window_size(1400, 900)
    return driver


def save_shot(driver, module, step):
    filename = OUT_DIR / f"{module}_{step}.png"
    driver.save_screenshot(str(filename))
    log(f"📸 Screenshot saved: {filename}")
    DOC.add_heading(f"{module} — {step}", level=2)
    DOC.add_picture(str(filename), width=Inches(6.5))
    return filename

def wait_for(driver, by, selector, timeout=30):
    log(f"driver: {driver}")
    log(f"by: {by}")
    log(f"selector: {selector}")
    log(f"timeout: {timeout}")
    return WebDriverWait(driver, timeout).until(EC.presence_of_element_located((by, selector)))


def click_link_in_table_by_text(driver, text):
    # Anchor text appears in the linked column; find it and click
    link = WebDriverWait(driver, 20).until(
        EC.element_to_be_clickable((By.XPATH, f"//table//a[normalize-space()='{text}']"))
    )
    log(f"Clicking link with text: {text}")
    link.click()


# [ADDED for dates] ----------
from datetime import timedelta

def _format_date_for_input(input_el, value_dt: datetime, fallback: str = None) -> str:
    """
    Given an <input> WebElement and a datetime, format the string that
    the browser control expects. Supports type="date" and "datetime-local".
    Falls back to 'YYYY-MM-DD HH:MM:SS' if no type found.
    """
    try:
        itype = (input_el.get_attribute("type") or "").lower()
    except Exception:
        itype = ""

    if isinstance(value_dt, datetime):
        if itype == "date":
            return value_dt.strftime("%Y-%m-%d")
        if itype == "datetime-local":
            # Most browsers require no seconds in datetime-local; trim to minutes
            return value_dt.strftime("%Y-%m-%dT%H:%M")
        # default textual datetime for typical server-side parsing
        return value_dt.strftime("%Y-%m-%d %H:%M:%S")

    # If value_dt is not a datetime, return as-is (string)
    return value_dt if value_dt is not None else (fallback or "")

def _set_input_value_js(driver, el, value_str: str):
    """Robustly set value of inputs like type=date/datetime-local via JS."""
    driver.execute_script("arguments[0].value = arguments[1];", el, value_str)

def try_fill_form_field(driver, name, value):
    """
    Wrap fill_form_field with a not-found guard so test continues
    even if a given field isn't on the page.
    """
    try:
        fill_form_field(driver, name, value)
    except Exception as e:
        log(f"[WARN] Could not fill field '{name}': {e}")

# ---------------------------

def _normalize_misc(value):
    if isinstance(value, str) and "miscellaneous" in value.lower():
        return "Miscellaneous"
    return value

def fill_form_field(driver, name, value):
    # Normalize "Miscellaneous" intent first
    value = _normalize_misc(value)

    # Try select first (for Category dropdowns)
    log(f"Filling field '{name}' = '{value}'")
    try:
        select_el = driver.find_element(By.NAME, name)
        if select_el.tag_name.lower() == "select":
            sel = Select(select_el)
            try:
                # Try exact (now normalized) match
                sel.select_by_visible_text(str(value))
                return
            except Exception:
                # Fallback: case-insensitive contains match
                target = str(value).lower()
                for opt in sel.options:
                    if target in (opt.text or "").lower():
                        sel.select_by_visible_text(opt.text)
                        return
            # If we get here, let it fall through to generic handling
    except Exception:
        pass

    # fallback: text/textarea/email/number/date inputs
    el = driver.find_element(By.NAME, name)
    tag = el.tag_name.lower()
    itype = (el.get_attribute("type") or "").lower()

    # If value is a datetime, format based on input type
    if isinstance(value, datetime):
        value = _format_date_for_input(el, value)

    # Special handling for date & datetime-local (clear/send_keys is flaky)
    if itype in ("date", "datetime-local"):
        vstr = str(value)
        if itype == "datetime-local" and "T" not in vstr:
            try:
                dt = datetime.strptime(vstr, "%Y-%m-%d %H:%M:%S")
                vstr = dt.strftime("%Y-%m-%dT%H:%M")
            except Exception:
                pass
        _set_input_value_js(driver, el, vstr)
        return

    if tag in ("input", "textarea"):
        try:
            el.clear()
        except Exception:
            from selenium.webdriver.common.keys import Keys
            el.send_keys(Keys.CONTROL, "a", Keys.DELETE)
        el.send_keys(str(value))
    else:
        # last resort
        el.send_keys(str(value))

def click_save(driver, board_id=None):
    """
    Click the Save button, optionally injecting a hidden BoardId and BoardMeetingId field first.
    """
    if board_id is not None:
        try:
            # --- Inject hidden BoardId before clicking Save ---
            driver.execute_script(
                """
                const form = document.querySelector('form');
                if (form && !form.querySelector('[name="BoardId"]')) {
                    const i = document.createElement('input');
                    i.type = 'hidden';
                    i.name = 'BoardId';
                    i.value = arguments[0];
                    form.appendChild(i);
                    console.log('Injected hidden BoardId:', i.value);
                }
                """,
                str(board_id),
            )
            html = driver.execute_script("return document.querySelector('form').outerHTML;")

            #important debug line if you are having trouble with boards
            #log(f"[DEBUG] Current form HTML:\n{html}")

            log(f"[INFO] Injected hidden BoardId={board_id} into form.")
        except Exception as ex:
            log(f"[WARN] Failed to inject BoardId hidden field: {ex}")

    # --- Proceed with your existing logic ---
    btn = wait_for(driver, By.XPATH, "//button[normalize-space()='Save']")
    try:
        btn.click()
        return
    except Exception as e:
        log(f"[WARN] Normal click failed: {e}. Trying scroll + JS click.")

    try:
        driver.execute_script("arguments[0].scrollIntoView({block:'center'});", btn)
        time.sleep(0.2)
        driver.execute_script("arguments[0].click();", btn)
    except Exception as e2:
        log(f"[ERROR] JS click also failed: {e2}")
        raise

def accept_confirm(driver, timeout=5):
    try:
        WebDriverWait(driver, timeout).until(EC.alert_is_present())
        driver.switch_to.alert.accept()
    except Exception:
        pass


def login_if_needed(driver):
    if not LOGIN_URL:
        return
    driver.get(LOGIN_URL)

    # Implement your login flow if you have auth, e.g.:
    # wait_for(driver, By.NAME, "email").send_keys(os.environ["APP_USER"])
    # driver.find_element(By.NAME, "password").send_keys(os.environ["APP_PASS"])
    # driver.find_element(By.XPATH, "//button[normalize-space()='Sign in']").click()
    # WebDriverWait(driver, 10).until(EC.url_contains("/home"))

    time.sleep(1)

    # Locate the Login button by visible text
    try:
        login_button = driver.find_element(By.XPATH, "//button[normalize-space()='Login']")
        login_button.click()
    except Exception as e:
        print("Login button not found:", e)

def pick_board(driver, value=None, text=None):
    select = Select(driver.find_element(By.ID, "f_board"))

    if value:
        select.select_by_value(str(value))
    elif text:
        select.select_by_visible_text(text)
    else:
        raise ValueError("Provide value= or text=")
    print("selected: ", str(value))

    # No page reload occurs; small pause lets JS settle
    time.sleep(0.5)

    # mabye this if ever needed
    # Stable wait: ensures the option is actually selected
    #WebDriverWait(driver, 5).until(
    #    lambda d: Select(d.find_element(By.ID, "f_board"))
    #                .first_selected_option
    #                .get_attribute("value") == str(value)
    #)

    # we found this does not work because
    # onchange does not trigger form submit
    #WebDriverWait(driver, 10).until(
    #    EC.staleness_of(driver.find_element(By.ID, "f_board"))
    #)

def create_record(driver, module, cfg, unique_suffix):
    # Always navigate directly to /module/create to avoid hunting the button
    driver.get(f"{BASE_URL}/{module}/create")

    #print("DEBUG NAVIGATING TO:", f"{BASE_URL}/{module}/create")
    #driver.get(f"{BASE_URL}/{module}/create")
    #print("DEBUG ACTUAL URL:", driver.current_url)
    #print("DEBUG PAGE SOURCE LENGTH:", len(driver.page_source))
    #open("debug_page.html", "w", encoding="utf-8").write(driver.page_source)
    #print("PAGE SOURCE:\n", driver.page_source)

    wait_for(driver, By.TAG_NAME, "form")
    save_shot(driver, module, "01_create_form")

    # Fill required fields
    data = cfg["create"].copy()

    # [ADDED for dates] Inject dates for announcements with proper types.
    # We'll send actual datetime objects; fill_form_field will format them per input type.
    print(f"module {module}")
    if module == "announcements":
        now = datetime.now()
        in_7 = now + timedelta(days=7)
        # Only set if your form uses these names; if names differ, add them here too.
        data.setdefault("StartDate", now)
        data.setdefault("EndDate", in_7)
    # Inject dates for events
    if module == "events":
        now = datetime.now().date()
        print(f"set event date {now}")
        data.setdefault("StartDate", now)       # required (NOT NULL)
        data.setdefault("EndDate", now) 
        # Optional: give it a default 1-hour window
        # data.setdefault("EndDate", now + timedelta(hours=1))
        # data.setdefault("AllDay", 0)  # only if your form expects it
    # Inject BoardId for boardsandcommitteemembers et al
    if module == "boardmeetings":
        now = datetime.now().date() 
        data.setdefault("Date", now)       # required (NOT NULL)
    if module == "ordinancesregulationsforms":
        print("set to Miscellaneous")
        data.setdefault("Category", "Miscellaneous")       # required (NOT NULL)

    # Make values unique so we can find the row later
    #for k in data:
    #    # Don't append uniqueness to datetime objects
    #    if isinstance(data[k], str):
    #        data[k] = f"{data[k]} {unique_suffix}"

    # Make values unique so we can find the row later — but only for free-text inputs.
    for k in list(data.keys()):
        el = get_form_element(driver, k)
        # Don't suffix email/url/select/date/datetime-local/number fields
        is_select = is_select_element(el)
        itype = (el.get_attribute("type") or "").lower() if el else ""
        print(f"itype: {itype}")
        iname = (el.get_attribute("name") or "").lower() if el else ""
        print(f"iname: {iname}")

        #print(f"is_select: {is_select}")
        #if iname in ("state", "title", "document", "agendaurl"):
        #    print(f"iname1")
        #if itype in ("email", "date", "datetime-local", "number"):
        #    print(f"iname2")
        #if "url" in iname.lower():
        #    print(f"iname3")

        #just a hack to get us by trouble fields
        non_free = is_select or iname in ("state", "title", "document", "agendaurl") or itype in ("email", "date", "datetime-local", "number") or "url" in iname.lower()

        if isinstance(data[k], str) and not non_free:
            data[k] = f"{data[k]} {unique_suffix}"

    for name, value in data.items():
        print(f"name: {name}, value: {value}")
        # [ADDED for dates] be tolerant if a field is not present
        try_fill_form_field(driver, name, value)

    # Tick Active where applicable
    try:
        active = driver.find_element(By.NAME, ACTIVE_CHECKBOX)
        if not active.is_selected():
            active.click()
    except Exception:
        pass

    log(f"[{module}] Starting CREATE step")
    save_shot(driver, module, "01b_create_form")
    if module == 'boardsandcommitteemembers' or module == 'boardmeetings' or module == 'boardschedulerules':
        click_save(driver, board_id=BOARD_ID)
    else:
        click_save(driver, board_id=None)
    log(f"[{module}] CREATE complete")

    if module == 'boardsandcommittees':
        set_board_id()

    # After save, most controllers redirect to index
    wait_for(driver, By.TAG_NAME, "table")
    save_shot(driver, module, "02_after_create_list")

    return data  # return the actual values used

def edit_record(driver, module, cfg, created_values):
    # Open list page
    driver.get(f"{BASE_URL}/{module}")
    wait_for(driver, By.TAG_NAME, "table")
    save_shot(driver, module, "03_list_before_edit")

    # Locate the row that contains our created record
    link_field = cfg["link_field"]
    link_text  = created_values[link_field]
    log(f"[{module}] Searching for row with text: {link_text}")

    # Find the table rows
    rows = driver.find_elements(By.CSS_SELECTOR, "table tbody tr")
    target_row = None
    for row in rows:
        if link_text.lower() in row.text.lower():
            target_row = row
            break

    if not target_row:
        raise Exception(f"Row containing '{link_text}' not found on {module} list page")

    # Within that row, find the Edit button and click it
    try:
        edit_btn = target_row.find_element(By.XPATH, ".//a[normalize-space()='Edit']")
        driver.execute_script("arguments[0].scrollIntoView({block:'center'});", edit_btn)
        time.sleep(0.15)
        driver.execute_script("arguments[0].click();", edit_btn)
    except Exception as e:
        raise Exception(f"Could not click Edit button for '{link_text}': {e}")

    # On the edit form now
    wait_for(driver, By.TAG_NAME, "form")
    save_shot(driver, module, "04_edit_form")

    # Append text to the link field to verify the update
    edit_append = cfg.get("edit_append", "edited")
    new_text = link_text + edit_append

    fill_form_field(driver, link_field, new_text)
    log(f"[{module}] Starting EDIT step")
    save_shot(driver, module, "04b_edit_form")
    if module == 'boardsandcommitteemembers' or module == 'boardmeetings' or module == 'boardschedulerules':
        click_save(driver, board_id=BOARD_ID)
    else:
        click_save(driver, board_id=None)
    log(f"[{module}] EDIT complete")

    wait_for(driver, By.TAG_NAME, "table")
    save_shot(driver, module, "05_after_edit_list")

    # Return updated value for later delete step
    created_values[link_field] = new_text
    return created_values

def delete_record(driver, module, cfg, created_values):
    if DELETE_ALL == False:
        return

    # Open list page
    driver.get(f"{BASE_URL}/{module}")
    wait_for(driver, By.TAG_NAME, "table")
    save_shot(driver, module, "06_list_before_delete")

    link_text = created_values[cfg["link_field"]]
    log(f"[{module}] Searching for row containing '{link_text}' for deletion")

    # Locate the matching row
    try:
        row = WebDriverWait(driver, 20).until(
            EC.presence_of_element_located(
                (By.XPATH, f"//table//tr[.//*[normalize-space()='{link_text}']]")
            )
        )
    except Exception as e:
        raise Exception(f"[{module}] Could not find row for '{link_text}': {e}")

    # Locate the Delete button within that row
    try:
        delete_btn = row.find_element(By.XPATH, ".//button[normalize-space()='Delete']")
    except Exception:
        # Sometimes it's rendered as <input type=submit value='Delete'> or <a>Delete</a>
        delete_btn = row.find_element(By.XPATH, ".//*[self::a or self::input or self::button][normalize-space()='Delete']")

    # Scroll into view and click via JS for reliability
    try:
        driver.execute_script("arguments[0].scrollIntoView({block:'center'});", delete_btn)
        time.sleep(0.15)
        driver.execute_script("arguments[0].click();", delete_btn)
    except Exception as e:
        raise Exception(f"[{module}] Could not click Delete button for '{link_text}': {e}")

    # Confirm deletion
    log(f"[{module}] Starting DELETE step")
    try:
        accept_confirm(driver)
    except Exception:
        # Some pages use modal confirmations instead of JS confirm()
        log(f"[WARN] No JS confirm found for {module}; assuming custom modal handled by JS")
    log(f"[{module}] DELETE complete")

    # Wait for list refresh & screenshot
    time.sleep(1.2)
    save_shot(driver, module, "07_after_delete_list")

def main():
    saved_configs = {}
    saved_created_values = {}
    driver = start_browser()
    try:
        login_if_needed(driver)

        # Read modules from list.txt
        log("Starting CRUD flow test.")
        topics_file = Path("list.txt")
        topics = [line.strip() for line in topics_file.read_text().splitlines() if line.strip()]
        log(f"Loaded modules from list.txt: {topics}")

        for module in topics:
            if module not in MODULES:
                print(f"[skip] No config for module: {module}")
                continue

            print(f"[run] {module}")
            # Go to list page and capture
            driver.get(f"{BASE_URL}/{module}")
            wait_for(driver, By.TAG_NAME, "body")
            save_shot(driver, module, "00_list")

            #click the dropdown to select correct Board 
            if module == 'boardmeetings':
                # NEW: pick board before filling the form
                board_value = BOARD_ID
                if board_value:
                   pick_board(driver, value=board_value)
                   wait_for(driver, By.TAG_NAME, "body")
                   save_shot(driver, module, "00B_list")

            unique_suffix = datetime.now().strftime("%H%M%S")
            cfg = MODULES[module]

            log(f"→ Running module: {module}")
            created_values = create_record(driver, module, cfg, unique_suffix)
            log(f"✔ Created record in {module}: {created_values}")
            created_values = edit_record(driver, module, cfg, created_values)
            log(f"✔ Edited record in {module}: {created_values}")
            saved_configs[module] = cfg
            saved_created_values[module] = created_values

            # boardsandcommittees record is created first, and if DELETE_ALL == False it never gets deleted, but if 
            # DELETE_ALL == True, it can't get deleted until the end because the other boardxxxxxx tables needs its BoardId,
            # so it is best to attempt to delete it after all the others. 
            # Same with boardmeetings
            log(f"before {module}")
            log(f"boolean {module != 'boardsandcommittees' and module != 'boardmeetings'}")
            if module != 'boardsandcommittees' and module != 'boardmeetings':
                log(f"inside {module}")
                delete_record(driver, module, cfg, created_values)
                log(f"✔ Deleted record in {module}")

            # Back to main menu (home)
            driver.get(f"{BASE_URL}")
            log(f"Returned to {BASE_URL} for {module}")
            save_shot(driver, module, "08_back_home")

        # now attempt to delete boardmeetings
        log(f"attempt meetings")
        driver.get(f"{BASE_URL}/boardmeetings")
        bm_cfg = saved_configs.get('boardmeetings')
        bm_values = saved_created_values.get('boardmeetings')
        log(f"{bm_cfg}")
        log(f"{bm_values}")
        if bm_cfg and bm_values:
            log(f"inside meetings")
            delete_record(driver, 'boardmeetings', bm_cfg, bm_values)
            log("✔ Deleted record in boardmeetings")
        else:
            log("[WARN] Missing saved data for boardmeetings")

        # now attempt to delete boardsandcommittees
        log(f"attempt boards")
        driver.get(f"{BASE_URL}/boardsandcommittees")
        bc_cfg = saved_configs.get('boardsandcommittees')
        bc_values = saved_created_values.get('boardsandcommittees')
        if bc_cfg and bc_values:
            log(f"inside boards")
            delete_record(driver, 'boardsandcommittees', bc_cfg, bc_values)
            log("✔ Deleted record in boardsandcommittees")
        else:
            log("[WARN] Missing saved data for boardsandcommittees")

        # Save the Word doc
        doc_name = f"crud_test_run_{RUN_STAMP}.docx"
        DOC.save(str(OUT_DIR / doc_name))
        print(f"Saved: {OUT_DIR / doc_name}")

    finally:
        log(f"All modules done. Word doc saved at {OUT_DIR}")
        driver.quit()


if __name__ == "__main__":
    main()
