<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;

use Exception;

class TownController
{
    /** GET /town */
    public function index(): void
    {
        $pdo = DB::pdo();
 
        $sql = "SELECT *, Active AS IsActive from Town";
        $rows = $pdo->query($sql)->fetchAll();

        $page_title = 'Town';
        require __DIR__ . '/../Views/town/list.php';
    }

    /** GET /town/create */
    public function create(): void
    {
        $pdo = DB::pdo();

        $record = [
            'Id' => null,
            'Name' => '',
            'Document' => ''
        ];
        $page_title = 'Add Town Info';
        require __DIR__ . '/../Views/town/form.php';
    }

    /** POST /town/store */
    public function store(): void
    {
        csrf_verify();
        $pdo = DB::pdo();
        $createdBy = \App\Helpers\current_username();

        $Name = $this->nullIfEmpty($_POST['Name'] ?? null);
        $Address1 = $this->nullIfEmpty($_POST['Address1'] ?? null);
        $Address2 = $this->nullIfEmpty($_POST['Address2'] ?? null);
        $Town = $this->nullIfEmpty($_POST['Town'] ?? null);
        $State = $this->nullIfEmpty($_POST['State'] ?? null);
        $Zip = $this->nullIfEmpty($_POST['Zip'] ?? null);
        $Phone1 = $this->nullIfEmpty($_POST['Phone1'] ?? null);
        $Phone2 = $this->nullIfEmpty($_POST['Phone2'] ?? null);
        $Email = $this->nullIfEmpty($_POST['Email'] ?? null);
        $Website = $this->nullIfEmpty($_POST['Website'] ?? null);
        $Remarks = $this->nullIfEmpty($_POST['Remarks'] ?? null);
        $Active = isset($_POST['IsActive']) ? 1 : 0;

        if (!$Name || !$Address1 || !$Town || !$State || !$Zip || !$Phone1) {
            http_response_code(400);
            echo "Name, full Address1 and Phone are required.";
            return;
        }

        $sql = "
            INSERT INTO town
            (Name, Address1, Address2, Town, State, Zip, Phone1, Phone2, Email, Website, Remarks, Active, CreatedAt, CreatedBy)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $Name,
            $Address1,
            $Address2,
            $Town,
            $State,
            $Zip,
            $$Phone1,
            $Phone2,
            $Email,
            $Website,
            $Remarks,
            $Active,
            $createdBy
        ]);

        $id = (int)$pdo->lastInsertId();
        audit('Town','create',$id,[
            $Name,
            $Address1,
            $Address2,
            $Town,
            $State,
            $Zip,
            $$Phone1,
            $Phone2,
            $Email,
            $Website,
            $Remarks,
            $createdBy
        ]);
        redirect('/town');
        //header("Location: /town");
        //exit;
    }

    /** GET /town/edit?id=123 */
    public function edit(): void
    {
        $id = (int)($_GET['id'] ?? 0);
        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }

        $pdo = DB::pdo();

        $stmt = $pdo->prepare("SELECT * FROM Town WHERE Id = ?");
        $stmt->execute([$id]);
        $model = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$model) {
            http_response_code(404);
            echo "Town record not found.";
            return;
        }

        // Normalize Active → IsActive for shared form
        if (!array_key_exists('IsActive', $model) && array_key_exists('Active', $model)) {
            $model['IsActive'] = (int)$model['Active'];
        }

        $errors = [];
        ob_start();
        $page_title = 'Edit Town Info';
        include __DIR__ . '/../Views/town/form.php';
        //return ob_get_clean();
    }

    /** POST /town/update */
    public function update(): void
    {
        csrf_verify();
        $id  = (int)($_POST['Id'] ?? 0);
        $pdo = DB::pdo();

        $sel = $pdo->prepare("SELECT * FROM Town WHERE Id=?");
        $sel->execute([$id]);
        $before = $sel->fetch() ?: [];

        $Name = $this->nullIfEmpty($_POST['Name'] ?? null);
        $Address1 = $this->nullIfEmpty($_POST['Address1'] ?? null);
        $Address2 = $this->nullIfEmpty($_POST['Address2'] ?? null);
        $Town = $this->nullIfEmpty($_POST['Town'] ?? null);
        $State = $this->nullIfEmpty($_POST['State'] ?? null);
        $Zip = $this->nullIfEmpty($_POST['Zip'] ?? null);
        $Phone1 = $this->nullIfEmpty($_POST['Phone1'] ?? null);
        $Phone2 = $this->nullIfEmpty($_POST['Phone2'] ?? null);
        $Email = $this->nullIfEmpty($_POST['Email'] ?? null);
        $Website = $this->nullIfEmpty($_POST['Website'] ?? null);
        $Remarks = $this->nullIfEmpty($_POST['Remarks'] ?? null);
        $Active = isset($_POST['IsActive']) ? 1 : 0;

        if (!$id || !$Name || !$Address1 || !$Phone1) {
            http_response_code(400);
            echo "Id, Name, complete Address and Phone1 are required.";
            return;
        }

        $sql = "
            UPDATE town
            SET Name=?, Address1=?, Address2=?, Town=?, State=?, Zip=?, Phone1=?, Phone2=?, Email=?, Website=?, Remarks=?, Active=?
            WHERE Id=?
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $Name,
            $Address1,
            $Address2, 
            $Town, 
            $State, 
            $Zip, 
            $Phone1, 
            $Phone2, 
            $Email, 
            $Website, 
            $Remarks, 
            $Active, 
            $id
        ]);

        $sel->execute([$id]);
        $after = $sel->fetch() ?: [];
        audit('Town','update',$id,['before'=>$before,'after'=>$after]);
        redirect('/town');
        //header("Location: /town");
        //exit;
    }

    /** POST /town/delete */
    public function delete(): void
    {
        $id = (int)($_POST['id'] ?? 0);
        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }

        $pdo = DB::pdo();
        $sel = $pdo->prepare("SELECT * FROM Town WHERE Id=?");
        $sel->execute([$id]);
        $row = $sel->fetch() ?: [];

        $stmt = $pdo->prepare("DELETE FROM town WHERE Id = ?");
        $stmt->execute([$id]);
        audit('Town','delete',$id,['before'=>$row]);

        redirect('/town');
        //header("Location: /town");
        //exit;
    }

    private function nullIfEmpty($v)
    {
        $v = is_string($v) ? trim($v) : $v;
        return ($v === '' ? null : $v);
    }
}
