<?php
namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;

class CommunityController
{
    public static function index(): string
    {
        $pdo  = DB::pdo();
        $q    = trim($_GET['q'] ?? '');
        $cat  = trim($_GET['category'] ?? '');

        $sql = "SELECT Id, Name, Category, Town, State, Phone, IsActive, CreatedAt
                  FROM Community
                 WHERE 1=1";
        $args = [];

        if ($cat !== '') { $sql .= " AND Category = ?"; $args[] = $cat; }
        if ($q   !== '') {
            $like = "%{$q}%";
            $sql .= " AND (Name LIKE ? OR Town LIKE ? OR Phone LIKE ?)";
            array_push($args, $like, $like, $like);
        }
        $sql .= " ORDER BY Category, Name";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($args);
        $rows = $stmt->fetchAll();

        // View expects $rows
        ob_start();
        $title = 'Community';
        include __DIR__ . '/../Views/community/list.php';
        return ob_get_clean();
    }

    public static function create(): string
    {
        $model = [
            'Id'=>0,
            'Name'=>'','Category'=>'',
            'Address1'=>'','Address2'=>'','Town'=>'','State'=>'','Zip'=>'',
            'Phone'=>'','Hours'=>'','Description'=>'','Notes'=>'',
            'IsActive'=>1,
        ];
        $errors = [];
        ob_start();
        $title = 'New Community Entry';
        include __DIR__ . '/../Views/community/form.php';
        return ob_get_clean();
    }

    public static function store(): void
    {
        csrf_verify();
        $pdo = DB::pdo();
        $createdBy = \App\Helpers\current_username();

        $data = [
            'Name'        => trim($_POST['Name'] ?? ''),
            'Category'    => trim($_POST['Category'] ?? ''),
            'Address1'    => trim($_POST['Address1'] ?? ''),
            'Address2'    => trim($_POST['Address2'] ?? ''),
            'Town'        => trim($_POST['Town'] ?? ''),
            'State'       => trim($_POST['State'] ?? ''),
            'Zip'         => trim($_POST['Zip'] ?? ''),
            'Phone'       => trim($_POST['Phone'] ?? ''),
            'Hours'       => trim($_POST['Hours'] ?? ''),
            'Description' => trim($_POST['Description'] ?? ''),
            'Notes'       => trim($_POST['Notes'] ?? ''),
            'IsActive'    => isset($_POST['IsActive']) ? 1 : 0,
        ];

        $errors = [];
        if ($data['Name'] === '')     $errors[] = 'Name is required.';
        if ($data['Category'] === '') $errors[] = 'Category is required.';

        if ($errors) {
            $model = array_merge(['Id'=>0], $data);
            ob_start();
            $title = 'New Community Entry';
            include __DIR__ . '/../Views/community/form.php';
            $html = ob_get_clean();
            echo $html;
            return;
        }

        $sql = "INSERT INTO Community
                  (Name,Category,Address1,Address2,Town,State,Zip,Phone,Hours,Description,Notes,IsActive,CreatedAt,CreatedBy)
                VALUES
                  (?,?,?,?,?,?,?,?,?,?,?, ?, NOW(3), ?)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $data['Name'],$data['Category'],$data['Address1'],$data['Address2'],$data['Town'],$data['State'],$data['Zip'],
            $data['Phone'],$data['Hours'],$data['Description'],$data['Notes'],$data['IsActive'],
            $createdBy
        ]);

        $id = (int)$pdo->lastInsertId();
        audit('Community','create',$id,['after'=>$data]);
        redirect('/community');
    }

    public static function edit(): string
    {
        $id = (int)($_GET['id'] ?? 0);
        $pdo = DB::pdo();
        $stmt = $pdo->prepare("SELECT * FROM Community WHERE Id=?");
        $stmt->execute([$id]);
        $model = $stmt->fetch();

        if (!$model) {
            http_response_code(404);
            return "Community record not found.";
        }

        $errors = [];
        ob_start();
        $title = 'Edit Community Entry';
        include __DIR__ . '/../Views/community/form.php';
        return ob_get_clean();
    }

    public static function update(): void
    {
        csrf_verify();
        $id  = (int)($_POST['Id'] ?? 0);
        $pdo = DB::pdo();

        $sel = $pdo->prepare("SELECT * FROM Community WHERE Id=?");
        $sel->execute([$id]);
        $before = $sel->fetch() ?: [];

        $data = [
            'Name'        => trim($_POST['Name'] ?? ''),
            'Category'    => trim($_POST['Category'] ?? ''),
            'Address1'    => trim($_POST['Address1'] ?? ''),
            'Address2'    => trim($_POST['Address2'] ?? ''),
            'Town'        => trim($_POST['Town'] ?? ''),
            'State'       => trim($_POST['State'] ?? ''),
            'Zip'         => trim($_POST['Zip'] ?? ''),
            'Phone'       => trim($_POST['Phone'] ?? ''),
            'Hours'       => trim($_POST['Hours'] ?? ''),
            'Description' => trim($_POST['Description'] ?? ''),
            'Notes'       => trim($_POST['Notes'] ?? ''),
            'IsActive'    => isset($_POST['IsActive']) ? 1 : 0,
        ];

        $errors = [];
        if ($data['Name'] === '')     $errors[] = 'Name is required.';
        if ($data['Category'] === '') $errors[] = 'Category is required.';

        if ($errors) {
            $model = array_merge(['Id'=>$id], $data);
            ob_start();
            $title = 'Edit Community Entry';
            include __DIR__ . '/../Views/community/form.php';
            $html = ob_get_clean();
            echo $html;
            return;
        }

        $sql = "UPDATE Community SET
                  Name=?, Category=?, Address1=?, Address2=?, Town=?, State=?, Zip=?, Phone=?,
                  Hours=?, Description=?, Notes=?, IsActive=?
                WHERE Id=?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $data['Name'],$data['Category'],$data['Address1'],$data['Address2'],$data['Town'],$data['State'],$data['Zip'],$data['Phone'],
            $data['Hours'],$data['Description'],$data['Notes'],$data['IsActive'],
            $id
        ]);

        audit('Community','update',$id,['before'=>$before,'after'=>$data]);
        redirect('/community');
    }

    public static function delete(): void
    {
        csrf_verify();
        $id = (int)($_POST['id'] ?? 0);
        $pdo = DB::pdo();

        $sel = $pdo->prepare("SELECT * FROM Community WHERE Id=?");
        $sel->execute([$id]);
        $row = $sel->fetch() ?: [];

        $stmt = $pdo->prepare("DELETE FROM Community WHERE Id=?");
        $stmt->execute([$id]);

        audit('Community','delete',$id,['before'=>$row]);
        redirect('/community');
    }
}
