<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;
use function App\Helpers\url;
use function App\Helpers\handleupload;

use Exception;

class BoardMeetingsController
{
    /** GET /boardmeetings */
    public static function index(): void
    {
        $pdo = DB::pdo();
    
        // ------------------------------------------------------
        // 1. Load board list (always alphabetically)
        // ------------------------------------------------------
        $boards = $pdo->query("
            SELECT Id, CommitteeName AS BoardName
            FROM boardsandcommittees
            ORDER BY CommitteeName
        ")->fetchAll(\PDO::FETCH_ASSOC);
    
        if (!$boards) {
            echo "No boards found.";
            return;
        }
    
        // ------------------------------------------------------
        // 2. Resolve selected board
        //    Priority:
        //      (a) ?board=...  (from dropdown)
        //      (b) session value
        //      (c) first board in list
        // ------------------------------------------------------
        if (isset($_GET['board'])) {
            $pboardid = (int)$_GET['board'];
            $_SESSION['board_context']['pboardid'] = $pboardid;

            $ptitle = '';
            foreach ($boards as $b) {
                if ((int)$b['Id'] === $pboardid) {
                    $ptitle = $b['BoardName'];
                    break;
                }
            }
            $_SESSION['board_context']['ptitle'] = $ptitle;

            \App\Helpers\log_info("BoardMinutes index using BoardId: " . $pboardid);
            \App\Helpers\log_info("BoardMinutes title: " . $ptitle);
            \App\Helpers\log_info("BoardMinutes isset");
        } else {
            $pboardid = $_SESSION['board_context']['pboardid'] ?? (int)$boards[0]['Id'];
            $ptitle = $_SESSION['board_context']['ptitle'] ?? $boards[0]['BoardName'];
            \App\Helpers\log_info("BoardMinutes index using BoardId: " . $pboardid);
            \App\Helpers\log_info("BoardMinutes title: " . $ptitle);
            \App\Helpers\log_info("BoardMinutes not isset");
        }
    
        // Safety fallback
        if (!$pboardid) {
            $pboardid = (int)$boards[0]['Id'];
        }
    
        \App\Helpers\log_info("BoardMinutes index using BoardId: " . $pboardid);
        \App\Helpers\log_info("BoardMinutes index using BoardId: " . $ptitle);
    
        // ------------------------------------------------------
        // 3. Pull ONLY rows for the selected board
        // ------------------------------------------------------
        $stmt = $pdo->prepare("
            SELECT 
                m.Id,
                m.BoardId AS BoardId,
                b.CommitteeName AS BoardName,
                DATE(m.MeetingDate) AS Date,
                m.AgendaUrl,
                m.ApprovedMinutesUrl,
                m.RecordingUrl
            FROM boardmeetingminutes m
            JOIN boardsandcommittees b ON b.Id = m.BoardId
            WHERE m.BoardId = ?
            ORDER BY m.MeetingDate DESC
        ");
        $stmt->execute([$pboardid]);
        $rows = $stmt->fetchAll(\PDO::FETCH_ASSOC);
    
        $page_title = 'Board Meeting Minutes';
    
        require __DIR__ . '/../Views/boardmeetings/list.php';
    }

    /** GET /boardmeetings/create */
    public static function create(): void
    {
        $pdo = DB::pdo();
        //$boards = $pdo->query("SELECT Id, CommitteeName as BoardName FROM boardsandcommittees ORDER BY CommitteeName")->fetchAll(\PDO::FETCH_ASSOC);

        $pboardid = $_SESSION['board_context']['pboardid'] ?? 0;
        $ptitle = $_SESSION['board_context']['ptitle'] ?? '';

        \App\Helpers\log_info('BoardMinutes create id:' . $pboardid);
        \App\Helpers\log_info('BoardMinutes create title:' . $ptitle);

        $model = [
            'pboardid' => $pboardid,
            'pboardid' => $pboardid,
            'ptitle' => $ptitle,
            'Id' => null,
            'BoardId' => $pboardid,
            'BoardName' => $ptitle,
            'Date' => '',
            'AgendaUrl' => '',
            'ApprovedMinutesUrl' => '',
            'RecordingUrl' => ''
        ];
        $page_title = 'Add Board Meeting Minutes';
        require __DIR__ . '/../Views/boardmeetings/form.php';
    }

    /** POST /boardmeetings/store */
    public static function store(): void
    {
        $pdo = DB::pdo();
        $createdBy = \App\Helpers\current_username();

        $pboardid = (int)($_GET['boardid'] ?? ($_SESSION['board_context']['pboardid'] ?? 0));
        $pparent  = $_GET['parent'] ?? ($_SESSION['board_context']['pparent'] ?? '');
        $ptitle   = $_GET['ptitle'] ?? ($_SESSION['board_context']['ptitle'] ?? '');

        $boardId = $pboardid;
        \App\Helpers\log_info('BoardMinutes id:' . $boardId);

        //for the Python crud tester
        if ($boardId == 0) {
            $boardId  = (int)($_POST['BoardId'] ?? 0);
            \App\Helpers\log_info('BoardMinutes id2:' . $boardId);
        }
        $boardId = 0;
        if ($boardId == 0) {
            $boardId  = (int)($_POST['BoardId'] ?? 0);
        }
        \App\Helpers\log_info('BoardMinutes mid:' . $boardId);

        //$sql = "Select CommitteeName as BoardName from BoardsAndCommittees where Id=?";
        //$stmt = $pdo->prepare($sql);
        //$stmt->execute([$boardId]);
        $BoardName = $ptitle;

        $MeetingDate = trim((string)($_POST['Date'] ?? '')); // yyyy-mm-dd from <input type="date">
        \App\Helpers\log_info('BoardMinutes date:' . $MeetingDate);
        $AgendaUrl = trim((string)($_POST['AgendaUrl'] ?? ''));
        $ApprovedMinutesUrl = trim((string)($_POST['ApprovedMinutesUrl'] ?? ''));
        $RecordingUrl = trim((string)($_POST['RecordingUrl'] ?? ''));

        if (!$boardId || !$MeetingDate) {
            http_response_code(400);
            echo "Board and Meeting Date are required.";
            return;
        }

        // Normalize date to full datetime (00:00:00)
        //$MeetingDateDT = $MeetingDate . ' 00:00:00';

        //Save file/url
        $parentTable = 'BoardMeetingMinutes';
        foreach (['AgendaUrl', 'ApprovedMinutesUrl', 'RecordingUrl'] as $field) {
            $inputName = 'attachment_' . $field;
            $upload = \App\Helpers\handleUpload($inputName, $parentTable);
            if ($upload) {
                if ($field == 'AgendaUrl') {
                    $AgendaUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                } elseif ($field == 'ApprovedMinutesUrl') {
                    $ApprovedMinutesUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                } else {
                    $RecordingUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                }
            }
        }

        $sql = "
            INSERT INTO boardmeetingminutes
            (BoardId, BoardName, MeetingDate, AgendaUrl, ApprovedMinutesUrl, RecordingUrl, CreatedAt, CreatedBy)
            VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $boardId,
            $BoardName,
            $MeetingDate,
            $AgendaUrl,
            $ApprovedMinutesUrl,
            $RecordingUrl,
            $createdBy
        ]);

        $id = (int)$pdo->lastInsertId();
        audit('BoardMeetingMinutes','create',$id,[
            $boardId,
            $BoardName,
            $MeetingDate,
            $AgendaUrl,
            $ApprovedMinutesUrl,
            $RecordingUrl,
            $createdBy
        ]);
        redirect('/boardmeetings');
    }

    /** GET /boardmeetings/edit?id=123 */
    public static function edit(): void
    {
        $id = (int)($_GET['id'] ?? 0);
        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }
        \App\Helpers\log_info('BoardMinutes edit1:' . $id);

        $pdo = DB::pdo();
        $boards = $pdo->query("SELECT Id, CommitteeName as BoardName FROM boardsandcommittees ORDER BY CommitteeName")->fetchAll(\PDO::FETCH_ASSOC);

        $_SESSION['board_context']['pid'] = (int)$id;

        $pboardid = (int)($_GET['boardid'] ?? ($_SESSION['board_context']['pboardid'] ?? 0));
        $pparent  = $_GET['parent'] ?? ($_SESSION['board_context']['pparent'] ?? '');
        $ptitle   = $_GET['ptitle'] ?? ($_SESSION['board_context']['ptitle'] ?? '');
        \App\Helpers\log_info('BoardMinutes edit2:' . $pboardid);
        \App\Helpers\log_info('BoardMinutes edit3:' . $pparent);
        \App\Helpers\log_info('BoardMinutes edit5:' . $ptitle);

        $stmt = $pdo->prepare("SELECT m.*, DATE(m.MeetingDate) as Date FROM boardmeetingminutes m WHERE m.Id = ?");
        $stmt->execute([$id]);
        $model = $stmt->fetch(\PDO::FETCH_ASSOC);
        \App\Helpers\log_info('BoardMinutes edit date:' . $model['Date']);
        \App\Helpers\log_info('BoardMinutes edit boardname:' . $model['BoardName']);
        $ptitle   = $model['BoardName'];
        \App\Helpers\log_info('BoardMinutes edit6:' . $ptitle);

        if (!$model) {
            echo "Data not found.";
            return;
        }

        // Convert MeetingDate to yyyy-mm-dd for <input type="date">
        //$record['MeetingDate'] = substr($record['MeetingDate'], 0, 10);

        $page_title = 'Edit Board Meeting Minutes';
        require __DIR__ . '/../Views/boardmeetings/form.php';
    }

    /** POST /boardmeetings/update */
    public static function update(): void
    {
        $pdo = DB::pdo();

        $id = (int)($_GET['id'] ?? ($_SESSION['board_context']['pid'] ?? 0));
        $pboardid = (int)($_GET['boardid'] ?? ($_SESSION['board_context']['pboardid'] ?? 0));
        $pparent  = $_GET['parent'] ?? ($_SESSION['board_context']['pparent'] ?? '');
        $ptitle   = $_GET['ptitle'] ?? ($_SESSION['board_context']['ptitle'] ?? '');

        $Id = $id;

        $sel = $pdo->prepare("SELECT * FROM BoardMeetingMinutes WHERE Id=?");
        $sel->execute([$id]);
        $before = $sel->fetch() ?: [];

        $BoardId = $pboardid;
        $BoardName = $ptitle;
        $MeetingDate = trim((string)($_POST['Date'] ?? ''));
        $AgendaUrl = trim((string)($_POST['AgendaUrl'] ?? ''));
        $ApprovedMinutesUrl = trim((string)($_POST['ApprovedMinutesUrl'] ?? ''));
        $RecordingUrl = trim((string)($_POST['RecordingUrl'] ?? ''));
        \App\Helpers\log_info('BoardMinutes update:' . $Id);
        \App\Helpers\log_info('BoardMinutes update:' . $BoardId);
        \App\Helpers\log_info('BoardMinutes update:' . $BoardName);
        \App\Helpers\log_info('BoardMinutes update:' . $MeetingDate);

        if (!$Id || !$BoardId || !$MeetingDate) {
            http_response_code(400);
            echo "Id, Board and Meeting Date are required.";
            return;
        }

        //$MeetingDateDT = $MeetingDate . ' 00:00:00';

        //Save file/url
        $parentTable = 'BoardMeetingMinutes';
        foreach (['AgendaUrl', 'ApprovedMinutesUrl', 'RecordingUrl'] as $field) {
            $inputName = 'attachment_' . $field;
            $upload = \App\Helpers\handleUpload($inputName, $parentTable);
            if ($upload) {
                if ($field == 'AgendaUrl') {
                    $AgendaUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                } elseif ($field == 'ApprovedMinutesUrl') {
                    $ApprovedMinutesUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                } else {
                    $RecordingUrl = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
                }
            }
        }

        $sql = "
            UPDATE boardmeetingminutes
            SET MeetingDate=?, AgendaUrl=?, ApprovedMinutesUrl=?, RecordingUrl=?
            WHERE Id=?
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $MeetingDate,
            $AgendaUrl,
            $ApprovedMinutesUrl,
            $RecordingUrl,
            $Id
        ]);

        $sel->execute([$id]);
        $after = $sel->fetch() ?: [];
        audit('BoardMeetings','update',$id,['before'=>$before,'after'=>$after]);
        redirect('/boardmeetings');
    }

    /** POST /boardmeetings/delete */
    public static function delete(): void
    {
        $pdo = DB::pdo();

        csrf_verify();
        $id = (int)($_POST['id'] ?? 0);

        if (!$id) {
            http_response_code(400);
            echo "Missing id.";
            return;
        }

        $sel = $pdo->prepare("SELECT * FROM BoardMeetingMinutes WHERE Id=?");
        $sel->execute([$id]);
        $row = $sel->fetch() ?: [];

        $stmt = $pdo->prepare("DELETE FROM boardmeetingminutes WHERE Id = ?");
        $stmt->execute([$id]);

        audit('BoardMeetings','delete',$id,['before'=>$row]);
        redirect('/boardmeetings');
    }

    private static function nullIfEmpty($v)
    {
        $v = is_string($v) ? trim($v) : $v;
        return ($v === '' ? null : $v);
    }
}
