<?php
namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;
use function App\Helpers\url;
use function App\Helpers\handleupload;

// config/bootstrap.php (or similar)
define('UPLOAD_ROOT', realpath(__DIR__ . '/../../uploads')); // parallel to /public
define('UPLOAD_BASE_URL', '/uploads');                        // how the browser reaches it

class AttachmentsController {

    //$config = include __DIR__ . '/../config.php';

    private static array $ALLOWED_PARENTS = ['Announcements','Community','Events','Resources','BoardsAndCommittees'];

    private static function validateParent(string $parentTable, string $parentTitle, int $parentId): void {
        if (!in_array($parentTable, self::$ALLOWED_PARENTS, true)) {
            http_response_code(400);
            echo "Invalid ParentTable";
            exit;
        }
        // For the incoming tables so far:
        // BoardsAndCommittees and Community is Name
        // Information is Role
        // All others is Title

        $pdo = DB::pdo();
        // Safe to interpolate because $parentTable is validated against an allow-list.

        // Determine which column to use based on the parent table
        switch ($parentTable) {
            case 'BoardsAndCommittees':
                $col = 'CommitteeName';
                break;

            case 'Community':
                $col = 'Name';
                break;

            case 'Information':
                $col = 'Role';
                break;

            default:
                $col = 'Title';
                break;
        }

        // Then safely build and prepare your statement
        $sql = "SELECT COUNT(1) FROM {$parentTable} WHERE Id=? AND {$col} = ?";
        $stmt = $pdo->prepare($sql);
        //$stmt = $pdo->prepare("SELECT COUNT(1) FROM {$parentTable} WHERE Id=? AND Title=?");

        \App\Helpers\log_info("Attachments.val Title: {$parentTitle}");
        $stmt->execute([$parentId, $parentTitle]);
        if (!$stmt->fetchColumn()) {
            http_response_code(404);
            echo "Parent not found";
            exit;
        }
    }

    public static function index(): string {
        $parentTable = (string)($_GET['parent'] ?? '');
        $parentId    = (int)($_GET['id'] ?? 0);
        $parentTitle    = (string)($_GET['ptitle'] ?? '');
        \App\Helpers\log_info("Attachments.index ['ParentTable'=>$parentTable,'ParentId'=>$parentId,'ParentTitle'=>$parentTitle]");
        self::validateParent($parentTable, $parentTitle, $parentId);

        $pdo = DB::pdo();
        $stmt = $pdo->prepare("SELECT * FROM Attachments WHERE ParentTable=? AND ParentId=? ORDER BY AttachmentId");
        $stmt->execute([$parentTable, $parentId]);
        $rows = $stmt->fetchAll();

        ob_start();
        $title = "Attachments for {$parentTable} '{$parentTitle}', Id={$parentId}";
        include __DIR__ . '/../Views/attachments/list.php';
        return ob_get_clean();
    }

    public static function create(): string {
        $parentTable = (string)($_GET['parent'] ?? '');
        $parentId    = (int)($_GET['id'] ?? 0);
        $parentTitle    = (string)($_GET['ptitle'] ?? '');
        \App\Helpers\log_info("Attachments.create ['ParentTable'=>$parentTable,'ParentId'=>$parentId,'ParentTitle'=>$parentTitle]");
        self::validateParent($parentTable, $parentTitle, $parentId);
        $resource = ['ParentTable'=>$parentTable,'ParentId'=>$parentId,'ParentTitle'=>$parentTitle];

        ob_start();
        $title = "Add Attachment";
        include __DIR__ . '/../Views/attachments/form.php';
        return ob_get_clean();
    }

public static function store(): void {
    csrf_verify();

    $pdo = DB::pdo();
    $createdBy = \App\Helpers\current_username();

    $parentTable = (string)($_POST['ParentTable'] ?? '');
    $parentId    = (int)($_POST['ParentId'] ?? 0);
    $parentTitle = (string)($_POST['ParentTitle'] ?? '');
    \App\Helpers\log_info("Attachments.store ['ParentTable'=>$parentTable,'ParentId'=>$parentId,'ParentTitle'=>$parentTitle],'CreatedBy'=>$createdBy]");
    self::validateParent($parentTable, $parentTitle, $parentId);

    $data = [
        'ParentTable' => $parentTable,
        'ParentId'    => $parentId,
        'ParentTitle' => $parentTitle,
        'Title'       => trim((string)($_POST['Title'] ?? '')),
        'Url'         => trim((string)($_POST['Url'] ?? '')) ?: null,
        'FilePath'    => null,
        'FileName'    => null,
        'IsActive'    => (isset($_POST['IsActive']) ? 1 : 0),
        'CreatedAt'   => gmdate('Y-m-d H:i:s'),
        'CreatedBy'   => $createdBy,
    ];

    //Save file/url
    $upload = \App\Helpers\handleupload('attachment', $parentTable);
    if ($upload) {
        $data['FilePath'] = $upload['FilePath'];
        $data['FileName'] = $upload['FileName'];

        // Build full URL using config PrependURL
        $data['Url'] = $GLOBALS['CONFIG']['app']['PrependURL'] . $upload['FilePath'];
        //$prepend = $config['app']['PrependURL'];
        //$data['Url'] = $prepend . $upload['FilePath'];
    }

    $sql = "INSERT INTO Attachments (ParentTable, ParentId, Title, Url, FilePath, FileName, IsActive, CreatedAt, CreatedBy)
            VALUES (?,?,?,?,?,?,?,NOW(),?)";

    $params = [
        $data['ParentTable'],
        $data['ParentId'],
        $data['Title'],
        $data['Url'],
        $data['FilePath'],
        $data['FileName'],
        $data['IsActive'],
        $data['CreatedBy'],
    ];

    \App\Helpers\log_info("Attachments.store SQL: {$sql}");
    \App\Helpers\log_info("Attachments.store params: " . json_encode($params, JSON_UNESCAPED_SLASHES));

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);

    $id = (int)$pdo->lastInsertId();
    audit('Attachments', 'create', $id, ['after' => $data]);

    redirect("/attachments?parent={$parentTable}&id={$parentId}&ptitle={$parentTitle}");
}

    public static function oldstore(): void {
        csrf_verify();

        $pdo = DB::pdo();
        $createdBy = \App\Helpers\current_username();

        $parentTable = (string)($_POST['ParentTable'] ?? '');
        $parentId    = (int)($_POST['ParentId'] ?? 0);
        $parentTitle    = (string)($_GET['ptitle'] ?? '');
        self::validateParent($parentTable, $parentTitle, $parentId);

        $data = [
            'ParentTable' => $parentTable,
            'ParentId'    => $parentId,
            'Title'       => trim((string)($_POST['Title'] ?? '')),
            'Url'         => trim((string)($_POST['Url'] ?? '')) ?: null,
            'FilePath'    => trim((string)($_POST['FilePath'] ?? '')) ?: null,
            'FileName'    => trim((string)($_POST['FileName'] ?? '')) ?: null,
            'IsActive'    => (isset($_POST['IsActive']) ? 1 : 0),
            'CreatedAt'   => gmdate('Y-m-d H:i:s'),
            'CreatedBy'   => $createdBy,
        ];

        // \App\Helpers\log_info('$_FILES contents: ' . print_r($_FILES, true));

        // ✅ BEGIN: File upload handling
        if (!empty($_FILES['attachment']['name'])) {
            $fileName  = basename($_FILES['attachment']['name']);

            // Physical path: D:\TownOfCushingME\uploads
            $uploadDir = dirname(__DIR__, 2) . '/uploads/';

            // Later for Linux
            //$uploadDir = str_replace('\\', '/', dirname(__DIR__, 2) . '/uploads/');

            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }

            $target = $uploadDir . $fileName;

            if (move_uploaded_file($_FILES['attachment']['tmp_name'], $target)) {
                // Web path (relative to localhost/TownOfCushingME/)
                $data['FilePath'] = 'uploads/' . $fileName;
                $data['FileName'] = $fileName;
            }
        }
        // ✅ END: File upload handling

        $sql = "INSERT INTO Attachments (ParentTable, ParentId, Title, Url, FilePath, FileName, IsActive, CreatedAt, CreatedBy)
                VALUES (?,?,?,?,?,?,?,NOW(),?)";

        $params = [
            $data['ParentTable'],
            $data['ParentId'],
            $data['Title'],
            $data['Url'],
            $data['FilePath'],
            $data['FileName'],
            $data['IsActive'],
            $data['CreatedBy'],
        ];

        \App\Helpers\log_info("Attachments.store SQL: {$sql}");
        \App\Helpers\log_info("Attachments.store params: " . json_encode($params, JSON_UNESCAPED_SLASHES));

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $id = (int)$pdo->lastInsertId();
        audit('Attachments', 'create', $id, ['after'=>$data]);

        redirect("/attachments?parent={$parentTable}&id={$parentId}");
    }

public static function delete(): void {
    csrf_verify();
    $id = (int)($_POST['Id'] ?? 0);
    $pdo = DB::pdo();

    // Fetch full row for audit + redirect context
    $sel = $pdo->prepare("SELECT * FROM Attachments WHERE AttachmentId=?");
    $sel->execute([$id]);
    $row = $sel->fetch();

    if (!$row) {
        redirect(url('/'));
        return;
    }

    // ✅ BEGIN: Delete physical file if it exists
    if (!empty($row['FilePath'])) {
        // Build absolute path from relative FilePath
        $absPath = dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $row['FilePath']);
        if (is_file($absPath)) {
            @unlink($absPath);
            \App\Helpers\log_info("Deleted attachment file: {$absPath}");
        } else {
            \App\Helpers\log_info("Attachment file not found for deletion: {$absPath}");
        }
    }
    // ✅ END: Delete physical file if it exists

    $del = $pdo->prepare("DELETE FROM Attachments WHERE AttachmentId=?");
    $del->execute([$id]);

    audit('Attachments', 'delete', $id, ['before' => $row]);

    redirect("/attachments?parent={$row['ParentTable']}&id={$row['ParentId']}");
}

    public static function olddelete(): void {
        csrf_verify();
        $id = (int)($_POST['Id'] ?? 0);
        $pdo = DB::pdo();

        // Fetch full row for audit + redirect context
        $sel = $pdo->prepare("SELECT * FROM Attachments WHERE AttachmentId=?");
        $sel->execute([$id]);
        $row = $sel->fetch();

        if (!$row) {
            redirect(url('/'));
            return;
        }

        $del = $pdo->prepare("DELETE FROM Attachments WHERE AttachmentId=?");
        $del->execute([$id]);

        audit('Attachments', 'delete', $id, ['before' => $row]);

        redirect("/attachments?parent={$row['ParentTable']}&id={$row['ParentId']}");
    }

public static function deactivate(): void {
    csrf_verify();
    $id = (int)($_POST['Id'] ?? 0);
    $pdo = DB::pdo();

    // Fetch existing row for audit + redirect context
    $sel = $pdo->prepare("SELECT * FROM Attachments WHERE AttachmentId=?");
    $sel->execute([$id]);
    $row = $sel->fetch();

    if (!$row) {
        redirect(url('/'));
        return;
    }

    // ✅ Mark inactive instead of deleting
    $upd = $pdo->prepare("UPDATE Attachments SET IsActive=0, UpdatedAt=NOW(), UpdatedBy=? WHERE AttachmentId=?");
    $upd->execute([\App\Helpers\current_username(), $id]);

    // Audit before/after change
    $after = $row;
    $after['IsActive'] = 0;
    $after['UpdatedBy'] = \App\Helpers\current_username();
    $after['UpdatedAt'] = gmdate('Y-m-d H:i:s');
    audit('Attachments', 'deactivate', $id, ['before' => $row, 'after' => $after]);

    redirect("/attachments?parent={$row['ParentTable']}&id={$row['ParentId']}");
}

}
